#INCLUDE P16F628A.inc
; Functions to output diagnostics via the TX line.
; All of Bank 2 (48 bytes) is used as a buffer. Due to timing considerations,
; no more than 48 bytes should be output in 2 successive messages. Output is
; started after TMR2 routine identifies end of second. This ensures output
; is completed in a period where possible interrupt conflicts are handled
; gracefully. However, it is common that new data is put into the transmit
; buffer while the previous data is still being transmitted. This could
; corrupt the previous data if there is too much data.
; There is only one diagnostic output function (TXhex). The other functions
; are housekeeping.
	    extern	INTunknown ; next interrupt in the chain
	    extern	EndInt ; exit from TX and TMR2 interrupts
	    global	TXinit ; set up initial pointers
	    global	TXtest ; initiate transmission if needed
	    global	INTtx ; service a TX interrupt
	    global	TXhex ; data to hex in TXbytes

; data and variables used by the UART TX routine
; -------------- common --------------
	    udata_shr

save	    RES     1   ; used to save STATUS when data copied to TXbytes

	    ; -- bank 1 --
bnk1C	    udata   0xE8 ; leave room for other bank 1 data
; variables
TXsent      RES     1 ; pointer to next byte to transmit
TXdata      RES     1 ; pointer to first byte in TXbytes to overwrite
TXstop	    RES	    1 ; end current transmit
saveFSR	    RES     1 ; needed during UART TX interrupt
length	    RES	    1 ; length of data to copy to TXbytes
from	    RES	    1 ; source of data to copy to TXbytes
TXflags	    RES	    1
TXnosp	    EQU	    0 ; suppress leading space

	    ; -- bank 2 --
; the assembler is unable to resolve some operands
;  unless these variables are explicitly defined
BufAddr	    EQU	    0x120	; start of bank 2
BufSiz	    EQU	    0x30	; size of bank 2

bk2mem      udata   BufAddr
TXbytes	    RES     BufSiz  ; the data buffer
	    
      	    code

; ----------- Set up TX pointers ---------
TXinit:
; REQUIRES RP0/RP1 point to bank 1 when called.
; Must be called once before any other TX routine.
; TXinit sets up the two pointers into the data buffer TXbytes.
; TXdata is the next empty byte. TXsent is the next byte to send.
; While TXdata = TXsent there is no data to transmit.
	MOVLW	TXbytes
	MOVWF	TXsent
	MOVWF	TXdata
	BSF	TXflags,TXnosp ; pretend last character sent was LF
	RETURN

; ----------- Put data in TXbytes waiting to be sent  ---------
TXhex:
; RP0 and IRP are saved and restored. RP1 must be zero.
; FSR points to a data variable in bank 0 or 1, W is the length.
; W may also contain in bits 3-7 a value from 1 to 26 causing the
; hex value to be preceded by a lower case letter a to z and an =
; sign.
; W | 7  6  5  4  3 | 2  1  0 | e.g. if W = 0x02 | (('g' & 0x1F)<<3)
;   |---------------|---------|      then g=xxxx is placed in TXbytes
;   | optional label| length  |      where xxxx is a hex value
; This routine converts the little endian value that FSR points to, into
; a big endian hex string in TXbytes with optional label. This may be
; preceded by a space if the previous sent was not a linefeed.
	MOVWF   save ; save length temporarily
       	SWAPF   STATUS,W    ; get state without disturbing it
	XORWF	save,W ; swap length and status
	XORWF	save,F ; status in save
	XORWF	save,W ; length in W
	BSF	STATUS,RP0 ; point to bank 1
	MOVWF	length ; now can save length in bank 1
	MOVF	FSR,W ; save source pointer
	MOVWF	from
	MOVLW	0x20 ; insert a space?
	BTFSS	TXflags,TXnosp   ; if last data was not linefeed
	CALL	TXnext ; insert a space before value
	BCF	TXflags,TXnosp
; check if preamble character
	MOVF	length,W
	ANDLW	0xF8 ; remove the length
	BTFSC	STATUS,Z
	GOTO	TX2hex ; no preamble
	XORWF	length,F ; clear preamble off stored length
	MOVWF	FSR ; use FSR as work space
	RRF	FSR,F ; move bits 3-7 to bits 0-4
	RRF	FSR,F
	RRF	FSR,W
	ANDLW	0x1F ; remove any spurious bits 5-7
	IORLW	0x60 ; make it an ASCII lower case letter
	CALL	TXnext
	MOVLW	"="
	CALL	TXnext
TX2hex:
	CALL	TXcalcptr ; get a pointer into the source string
	SWAPF   INDF,W	; we want bits 4-7 in bits 0-3 first
	CALL	TXcvhex ; converts 0-3 to hex and puts into TXbytes
	CALL	TXcalcptr ; get the pointer again
	MOVF	INDF,W	; now bits 0-3
	CALL	TXcvhex
	DECFSZ	length,F
	GOTO    TX2hex
; all done - restore caller's status and return
TXexit:
	SWAPF	save,W
	MOVWF	STATUS
	RETURN
TXcalcptr:
; make FSR a pointer into the source
	DECF	from,W ; length from end to 1 but we need end-1 to 0
	ADDWF	length,W ; so move 'from' back 1 and add length
	MOVWF	FSR
	BCF	STATUS,IRP ; bank 0 or 1 for indirect source
	RETURN
TXcvhex:
; oonvert lower 4 bits of W into hex character in TXbytes
	ANDLW   0x0F	; remove the other nibble
	ADDLW   0x06    	; DC overflow if > 9
	BTFSC   STATUS,DC	
	ADDLW   0x07	; this is the gap between characters 9 and A
	ADDLW   0x2A	; convert to an ASCII hex character 0-9,A-F
; fall through here to put hex char into TXbytes
TXnext:
	MOVWF	FSR ; save the character temoprarily in FSR
	MOVF    TXdata,W    ; address of next available byte to W
	XORWF	FSR,W	; swap the character and the address
	XORWF	FSR,F
	XORWF	FSR,W	; pointer in FSR, data in W
	BSF	STATUS,IRP ; bank 2 for indirect destination
	MOVWF	INDF ; put the character into TXbytes buffer
	INCF	FSR,W ; now update pointer
	XORLW	TXbytes+BufSiz	; test end of buffer
	BTFSC   STATUS,Z
	MOVLW	(BufAddr+BufSiz) ^ BufAddr 
	XORLW	TXbytes+BufSiz	    ; circular buffer
; W now has pointer to next available byte in TXbytes
	MOVWF	TXdata
	RETURN

; ----------- UART TX interrupt process ---------
; This interrupt only happens while TXIE is enabled. TXIE is enabled after
; the one second TMR2 routine (see TXtest). This routine transmits the data
; until the data identified by TXtest has been sent, then unsets TXIE.

INTtx:
; first test if it is a UART TX interrupt
	BTFSS  PIR1,TXIF ; skip if this is UART TX interrupt
	GOTO   INTunknown	; not TX, next test
; yes it is
	MOVLW	1 << RP0 | 1 << IRP  ; variables in bank 1, buffer bank 2
	MOVWF	STATUS
	MOVF	FSR,W	; save FSR (not required for other
	MOVWF	saveFSR	; interrupts, so not saved otherwise)
;
	MOVF	TXsent,W    ; address of next byte to W
	MOVWF   FSR	    ; address to FSR
	INCF	TXsent,W	    ; next address
	XORLW	TXbytes+BufSiz	; test end of buffer
	BTFSC   STATUS,Z
	MOVLW	(BufAddr+BufSiz) ^ BufAddr
	XORLW	TXbytes+BufSiz	    ; circular buffer
	MOVWF	TXsent ; update the pointer
	XORWF	TXstop,W    ; last byte in the current transmission?
	BTFSS   STATUS,Z
	GOTO	Usend	; no, just send the character
	BCF     PIE1,TXIE   ; yes, clear the interrupt enable flag
Usend:
	BCF     STATUS,RP0
	MOVF	INDF,W ; IRP=1 set before, get char from bank 2
	MOVWF   TXREG	; and send it
;
; restore FSR then use the general interrupt exit to restore the rest
;
	BSF     STATUS,RP0  ; back to bank 1
	MOVF    saveFSR,W
	MOVWF   FSR
	GOTO	EndInt
	
; ------------ test if TX should start ----------

TXtest:
; Chained from TMR2 interrupt at end of second.
; TXtest compares the pointers, if not equal initiates transmission.
; This is written as inline code to save call stack space and minimise
; instruction count.
	BSF	STATUS,RP0 ; point to bank 1
	MOVF	TXsent,W    ; is there data to transmit
	XORWF	TXdata,W
	BTFSC   STATUS,Z
	GOTO	EndInt	    ; nothing to transmit
; yes, add a linefeed
	MOVF	FSR,W	; save FSR (not required for other
	MOVWF	saveFSR	; interrupts, so not saved otherwise)
	MOVF	TXdata,W    ; address of next byte to W
	MOVWF   FSR	    ; address to FSR
	INCF	TXdata,W	    ; next address
	XORLW	TXbytes+BufSiz	; test end of buffer
	BTFSC   STATUS,Z
	MOVLW	(BufAddr+BufSiz) ^ BufAddr
	XORLW	TXbytes+BufSiz	    ; circular buffer
	MOVWF	TXdata
	MOVWF	TXstop    ; stop this transmission here
	MOVLW	0x0A ; line feed (LF)
	BSF	STATUS,IRP ; bank 2 for indirect
	MOVWF	INDF ; data into buffer
	BSF	TXflags,TXnosp ; next data is start of line so no space needed
	BSF     PIE1,TXIE   ; set the TX interrupt enable flag
	MOVF	saveFSR,W	; restore FSR
	MOVWF	FSR
	GOTO	EndInt	    ; standard interrupt exit
	
	end
